/* PROJECT-LEVEL WOMEN'S EMPOWERMENT IN AGRICULTURE INDEX (PRO-WEAI) v1.1
	CALCULATE PRO-WEAI DO-FILE
	UDPATED MAY 2020
	CONTACT: IFPRI-WEAI@CGIAR.ORG
	
	THIS DO-FILE CALCULATES PRO-WEAI FROM THE 12 PRO-WEAI INDICATORS. 
	NOTE: THIS DO-FILE IS BASED ON A DRAFT VERSION OF THE PRO-WEAI AND IS SUBJECT TO CHANGE. 
	
	UPDATES:
	(v1.1) May 18, 2020:	Changed from float to double: lines 107, 108, 133, 328, 329, 338-341, 344-347 
	
	(v1.2) May 25, 2022: Updated to allow for index construction using either 10 indicators or 12 indicators (including optional indicators of group_inf and respect) 
	
	UPDATES:
	line 482-484 added to calculate individual level gender parity indices
	line 506-510 added to calculate individual level empowerment indices
	line 573-581 added to calculate individual level additional empowerment indices
	line 586 added to calculate individual level ProWEAI
	line 606 onwards: added additional output tables to show results from original calculations and alternative calculations.
	line 608 summarizes community level indices, 
	line 610 summarizes individual level indices for women (for validation), 
	line 614 summarizes community level indices and line 616 summarizes individual level indices for men (for validation)
	line 621-624 added to calculate standard errors of key variables for all valid women's records
	line 628-630 added to calculate standard errors of gender parity variables*/

	
********************************************************************************


clear all
set more off

********************************************************************************
					*** OPEN FILE AND LOG *** 
********************************************************************************

** Set working directory
cd "[working directory file path]"

** Store project name
global projectname "[projectname]"

** Open log
capture log close
log using "${projectname}_proweai_calculate_log", text replace 

** Open file with pro-WEAI indicators
use "${projectname}_proweai_dataprep", clear


// SETTINGS

global indicators_12 = 0 // SET THIS EQAUL TO 1 IF PROJECT HAS 12 INDICATORS - Please remove /* */ from around the optional indicators in line 60
global selfeff_missing = 0 // SET THIS EQAUL TO 1 IF PROJECT DOES NOT HAVE SELF EFFICACY INDICATOR


********************************************************************************
					*** CALCULATE PRO-WEAI *** 
********************************************************************************

**************************************************
******   THREE DOMAINS EMPOWERMENT (3DE)   *******
**************************************************

// So far all_indicators were defined so 1 identifies adequate. //
// Now we transform indicators so 1 identifies inadequate. //

** If using all 12 indicators, remove "/*  */" from around group_inf and respect variables in line below. **

foreach var in autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember /*group_inf respect*/ {
	rename `var' `var'_ndepr
	gen `var'=1 if `var'_ndepr==0
	replace `var'=0 if `var'_ndepr==1
	}

*We are now starting with 0-1 variables where 1 means that the person is inadequate in that indicator. 

gen weight=1 // Note: =1 if unweighted; otherwise, assign variable containing individual sampling weights

	
tempfile depr_indicators
save `depr_indicators', replace 


*****************************************************************************
********  Create a local variable with all CORE indicators varlist_emp ******
*****************************************************************************
if $selfeff_missing == 1 {
#delimit;
local varlist_emp autonomy_inc never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember group_inf respect;

gen sample1=(autonomy_inc~=. & never_violence~=. & feelinputdecagr~=. & assetownership~=. & credit_accdec~=. & incomecontrol~=. & work_balance~=. & mobility~=. & groupmember~=. & group_inf~=. & respect~=.);
#delimit cr

} // projects that don't have self eff indicator but does include Influential groups and Respect Among HH Members


if $indicators_12 == 1 {
#delimit;
local varlist_emp autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember group_inf respect;

gen sample1=(autonomy_inc~=. & selfeff~=. & never_violence~=. & feelinputdecagr~=. & assetownership~=. & credit_accdec~=. & incomecontrol~=. & work_balance~=. & mobility~=. & groupmember~=. & group_inf~=. & respect~=.);
#delimit cr
}

else {
#delimit;
local varlist_emp autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember;

gen sample1=(autonomy_inc~=. & selfeff~=. & never_violence~=. & feelinputdecagr~=. & assetownership~=. & credit_accdec~=. & incomecontrol~=. & work_balance~=. & mobility~=. & groupmember~=.);
#delimit cr
}

**********************************************************************************
**** Define the CORE weights. Weights sum to 1 (not to the number of indicators)**
**********************************************************************************
*********** Create a loop for the variables with the same weight *****************
**********************************************************************************

*We now create the indicatorsí weights.*

if $indicators_12 == 1 {
foreach var in autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember group_inf respect {
	gen double w_`var'=1/12
}
} // projects that include optional indicators

else {
foreach var in autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember {
	gen double w_`var'=1/10
	}
	}

*******************************************************************
*********     Define the weighted inadequacy g0* matrix       ****
*******************************************************************

// WE FOCUSED ON THE MEASURE OF INADEQUACIES (DISEMPOWERMENT). //

foreach var in `varlist_emp'{
	gen double wg0_`var'= `var'*w_`var'
}

******************************************************************************
*********** Compute the frequency of missing values for indicator ************
******************************************************************************

foreach var in `varlist_emp' {

gen `var'_miss=1 if `var'==.
replace `var'_miss=0 if `var'!=.
}

sum *_miss

********************************************************************************
*************   Define the (weighted) inadequacy count vector "ci" ************
********************************************************************************

egen double ci=rsum(wg0_*)
label variable ci "Inadequacy Count"

egen n_missing=rowmiss(wg0_*)
label variable n_missing "Number of missing variables by individual"
gen missing=(n_missing>0)
label variable missing "Individual with missing variables"

*** Check sample drop due to missing values
tab missing
*drop if missing

***********************************************************************
***** Create the identification vector (inadequate/adequate) ***********
***** and compute individual average of inadequacy    *****************
***********************************************************************

egen total_w=total(weight) if missing==0

// FIRST, WE COMPUTED THE DISEMPOWERMENT IN AGRICULTURE INDEX (DAI). //
// AFTERWARDS, WE COMPUTE THE EMPOWERMENT IN AGRICULTURE INDEX (HERE CALLED EAI): EAI = 1 - DAI. //

*These are now percentages - this creates DAI by each percentage. 

forvalues x=1(1)100 { // FOR EACH POSSIBLE CUTOFF X BETWEEN 1% AND 100% //
	gen ch_`x'p=(ci>(`x'/100))	// WE CREATE A VARIABLE THAT IDENTIFIES THE DISEMPOWERED INDIVIDUALS (THOSE WHO HAVE AN INADEQUACY SCORE HIGHER THE X%). //
	replace ch_`x'p=. if missing==1
	gen a_`x'p=(ci) if ch_`x'p==1 // WE COMPUTE THE INDIVIDUAL INADEQUACY OF THOSE WHO ARE DISEMPOWERED. //
	replace a_`x'p=. if missing==1
	egen DAI_`x'p= total(ci*ch_`x'p*weight/total_w) // WE COMPUTE THE DISEMPOWERMENT INDEX (FOR EACH POSSIBLE CUTOFF X) //
	gen EAI_`x'p=1-DAI_`x'p // THEN, WE OBTAIN THE EMPOWERMENT INDEX. //
	label var ch_`x'p "Condition of disempowerment  k=`x'%"
	label var a_`x'p "Individual Average inadequacy  k=`x'"
	label var DAI_`x'p "National Disempowerment Index k=`x'%"
	label var EAI_`x'p "Combined Empowerment Index k=`x'%"
	}

// PLEASE NOTE THAT THESE ARE NOT YET THE 5DE. SO FAR WE ARE STILL LOOKING AT WOMEN AND MEN TOGETHER AND WE HAVE NOT YET DEFINED THE CUTOFF WE WANT TO USE. //

summarize ch_* a_* DAI_* EAI_* [aw=weight]

************************************************************************
******* Compute raw headcounts        **********************************
************************************************************************

foreach var in `varlist_emp' {
gen `var'_raw=(`var')
replace `var'_raw=. if missing==1
}

su *_raw  [iw=weight]

***********************************************************************************
*********** Compute Censored headcount by subgroups (gender or region etc)   ******
***********************************************************************************

// NOW WE DEFINE THE CUTOFF THAT WE WANT TO USE AND WE START LOOKING AT WOMEN AND MEN SEPARATELY //

* Please define in the first line your cutoff, the example shows k=20 is 20% of the variables
* In the second line replace with the name of the categorical variable (the variable name by which censored headcount is to be generated for the variables)
* that represents the different subgroups. 
* The subgroup variable must be coded in consecutive natural numbers starting in 1

pause

if $indicators_12 == 1 {
local k=25
} // projects that have 12 indicators

else {
local k=20
}

gen gender=sex

local r="gender"

foreach var in `varlist_emp' {
gen `var'_CH_`k'p=(`var'==1 & ch_`k'==1)
replace `var'_CH_`k'p=. if missing==1
}

summarize *_CH_`k'p [iw=weight]

*****************************************************************************************
*****************************************************************************************
**** Define decomposition rule (country, sex)
**** We keep the information of the weighted population before reducing the sample to only 
**** those cases with information in all the indicators considered

egen total_b = total(weight)
label var total_b "Total Population Before Sample Drop"
egen pop_shr_before = total(weight/total_b), by(`r')
label var pop_shr_before "Weighted Population Share of Each `r' before Sample Reduction"
gen temp=1 // We generate this variable for counting observations
egen sample_r_before = total(temp), by(`r')
label var sample_r_before "Sample Size of each `r' before Sample Reduction"

egen pop_shr_after = total(weight/total_w) if miss==0, by(`r')
label var pop_shr_after "Weighted Population Share of Each `r' after Sample Reduction"
egen sample_r_after = total(temp) if missing==0, by(`r')
label var sample_r_after  "Sample Size of Each `r' after Sample Reduction"
gen sample_lost_ratio= sample_r_after/sample_r_before
label var sample_lost_ratio  "Relative size of the final sample after reduction in each `r'"

************************************************************************************
**** Collapsing ********************************************************************
* So far, our database has individual level data, if we want to aggregate
* at any level, we use the command ìcollapseî. Collapse calculates weighted 
* averages at the  level defined by the user (gender), if the option "by(gender)"
* is not specified, the observations are aggregated at the national level.
* Before collapse, save your results using the following command
*************************************************************************************

tempfile PROWEAI_individual_indices
save `PROWEAI_individual_indices', replace 

// SAVES, FOR EACH COUNTRY, A DATASET WITH INDIVIDUAL DATA. //
// THIS DATASET INCLUDES INDIVIDUAL INADEQUACY COUNT, VARIABLES THAT IDENTIFY DISEMPOWERED FOR EACH CUTOFF AND VALUE OF DAI AND EAI FOR EACH CUTOFF. //
// PLEASE REMEMBER THAT DAI AND EAI WERE COMPUTED CONSIDERING WOMEN AND MEN TOGETHER. //
 
* You can use also the commands preserve before the command ìcollapseî and restore just after
* preserve

// NOW WE COMPUTE RELEVANT VARIABLES BY GENDER. //

egen pop_shr = total(weight/total_w) if miss==0, by(`r')

* collapse
* The following command will "collapse" our individual results according to the subgroup previously defined. 
//pause
collapse ch_* a_* *_CH_`k'p *_raw w_* EAI_* *_miss missing DAI_* pop_shr* sample_r_* sample_lost_ratio [aw=weight],by(`r')

* You have already calculated the national DAI. With the following lines you will calculate the 
* DAI for every region using the formulation M0=H*A obtained after collapsing the dataset.

// ATTENTION: DAI AND EAI REFER TO NATIONAL FIGURES. M0 AND EA REFER TO GENDER FIGURES. //

forvalues x=1(1)100 {
	gen M0_`x'p=ch_`x'p*a_`x'p
	label var M0_`x'p "Population Subgroup DAI k=`x'%"
	gen EA_`x'p=1-M0_`x'p
	label var EA_`x'p "Population Subgroup EAI k=`x'%"
	ren ch_`x'p H_`x'p
	label var H_`x'p "Population Subgroup Multidimensional Headcount Ratio k=`x'%"
	ren a_`x'p A_`x'p
	label var A_`x'p "Population Subgroup Average Inadequacy k=`x'%"
	label var DAI_`x'p "National DAI k=`x'%"
	}

foreach var in `varlist_emp' {
	gen `var'_cont_`k'_EAI=((`var'_CH_`k'p* w_`var')/ EA_`k'p)
	label var `var'_cont_`k'_EAI "Decomposed Contribution of `var' to the total Empowerment k=`k'"

	gen `var'_cont_`k'_DAI=((`var'_CH_`k'p* w_`var')/ M0_`k'p)
	label var `var'_cont_`k'_DAI "Decomposed Contribution of `var' to the total Disempowerment k=`k'"

	label var  `var'_CH_`k'p  "Decomposed Censored Headcount `var' k=`k'"
	label var  `var'_raw  "Decomposed Raw Headcount `var'"
	label var  `var'_miss  "Decomposed Missing values `var'"
	}

label variable pop_shr "Population Share"
gen cont_group_`k'=M0_`k'p/DAI_`k'p*pop_shr
label variable cont_group_`k' "Decomposed Contribution"

gen cont_subgroup_DAI_`k'=M0_`k'p/DAI_`k'p*pop_shr_after
label variable cont_subgroup_DAI_`k' "Population Subgroup Contribution to DAI"

gen cont_subgroup_EAI_`k'=EA_`k'p/EAI_`k'p*pop_shr_after
label variable cont_subgroup_EAI_`k' "Population Subgroup Contribution to EAI"

capture decode `r', gen(level)
drop `r'

gen sex=_n
label define sex_lab 1 "Male" 2 "Female"
label values sex sex_lab

tempfile PROWEAI_results_`r'
save `PROWEAI_results_`r'', replace 

// THE DATASETS INCLUDE THE DISEMPOWERMENT FIGURES FOR ALL CUTOFFS BETWEEN 1% AND 100%. WHEN EXTRACTING THE INFO WE FOCUS ON THE RELEVANT CUTOFF. //
// PLEASE SEE BELOW HOW TO EXTRACT RELEVANT INFORMATION FOR CUTOFF 20%. //


*********************************************
*******   GENDER PARITY INDEX (GPI)   *******
*********************************************

use `depr_indicators', clear

** Focus on male and female households

sort hhid sex
bys hhid: gen i=_n
bys hhid: egen n=max(i)

*tab hh_type n, miss
drop if n==1

*****************************************************************************
********  Create a local variable with all CORE indicators varlist_emp ******
*****************************************************************************

* Projects with 11 indicators

if $indicators_12 == 1 {
#delimit;
local varlist_5do autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember group_inf respect;

gen sample5do=(autonomy_inc~=. & selfeff~=. & never_violence~=. & feelinputdecagr~=. & assetownership~=. & credit_accdec~=. & incomecontrol~=. & work_balance~=. & mobility~=. & groupmember~=. & group_inf~=. & respect~=.);
#delimit cr

******************************
**** Define the weights.  ****
******************************

foreach var in autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember group_inf respect {
	gen double w_`var'=1/12
	}
	} // projects that have 12 indicators


* All other projects
else {
#delimit;
local varlist_5do autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember ;

gen sample5do=(autonomy_inc~=. & selfeff~=. & never_violence~=. & feelinputdecagr~=. & assetownership~=. & credit_accdec~=. & incomecontrol~=. & work_balance~=. & mobility~=. & groupmember~=.);
#delimit cr

******************************
**** Define the weights.  ****
******************************

foreach var in autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember {
	gen double w_`var'=1/10
	}
	}
	
**********************************************************
*********     Define the weigted inadequacy g0*      ****
**********************************************************

foreach var in `varlist_5do'{
	gen double wg0_`var'= `var'*w_`var'
	}

********************************************************************************
*************   Define the (weighted) inadequacy count vector "ci" ************
********************************************************************************

egen double ci=rsum(wg0_*)
replace ci=round(ci, 0.0001)
replace ci = . if sample5do==0

label variable ci "Inadequacy Count without Parity"

********************************************
*** Compute censored inadequacy scores  ***
********************************************

bys hhid: gen double w_ci_id=ci if sex==2 
bys hhid: gen double m_ci_id=ci if sex==1 
bys hhid: egen double W_ci=max(w_ci_id)
bys hhid: egen double M_ci=max(m_ci_id)
drop w_ci_id m_ci_id

bys hhid: gen double W_cen_ci=W_ci
bys hhid:replace W_cen_ci=`k'/100 if W_cen_ci<=(`k'/100) & W_cen_ci!=.
bys hhid: gen double M_cen_ci=M_ci
bys hhid:replace M_cen_ci=`k'/100 if M_cen_ci<=(`k'/100) & M_cen_ci!=.

******************************************************
*** Identify inadequate in terms of gender parity  ***
******************************************************

bys hhid: gen ci_above=(W_cen_ci>M_cen_ci) 
bys hhid: replace ci_above=. if W_cen_ci==.|M_cen_ci==.
label var ci_above "Equals 1 if individual lives in MF hh where the depr score of the woman is higher than the man - EI 1"

************************************
*** Compute Gender Parity Index  ***
************************************

** Full sample
gen female=(sex==2 & ci_above!=.)
egen women_n=total(female)
egen women_wt=total(female*weight)
drop female

* Verification
gen women_i=(sex==2 & M_cen_ci!=. & W_cen_ci!=.)
egen women_wt2=total(women_i*weight)
tab women_wt women_wt2, miss
drop women_i women_wt2

** Headcount ratio of inadequate women
gen inadequate=(ci_above==1 & sex==2)
egen inadequate_n = total(inadequate)
gen H=inadequate_n/women_n // Considering unweighted sample //
egen inadequate_wt = total(inadequate*weight) 
gen H_wt=inadequate_wt/women_wt // Considering weighted sample //

*Verification
gen inadequate_i=(M_cen_ci<W_cen_ci & sex==2 & M_cen_ci!=. & W_cen_ci!=.)
egen inadequate_wt2=total(inadequate_i*weight)
tab inadequate_wt inadequate_wt2, miss
drop inadequate_i inadequate_wt2

** Computation of normalized gap
qui gen ci_gap=(W_cen_ci-M_cen_ci)/(1-M_cen_ci) if ci_above==1 & sex==2 
egen ci_gap_sum = total(ci_gap*weight)
gen ci_average=ci_gap_sum/inadequate_wt

** Computation of GPI
gen H_GPI=inadequate_wt/women_wt
gen P1=H_GPI*ci_average
gen GPI=1-P1

//Individual level GPI computation //added
g IndP1=ci_above * ci_gap
replace IndP1=0 if ci_above==0
g IndGPI=1-IndP1

**************************
*** Summarize results  ***
**************************

sum H_GPI ci_average P1 GPI 
count if sex==2
tab women_n women_wt
capture drop _merge

tempfile PROWEAI_results_GPI
save `PROWEAI_results_GPI', replace 

*******************************************
*** Merge Individual GPI and Gender PROWEAI datasets  ***
*******************************************

use `PROWEAI_individual_indices.dta', clear
	capture drop _merge
	
	//creating individual level 3DE //added
	gen IndM0_25p=ch_25p*a_25p
	replace IndM0_25p=0 if ch_25p==0 //replace disempowerment score equal to 0 for empowered individuals
	gen IndEA_25p=1-IndM0_25p
	ren ch_25p IndH_25p
	ren a_25p IndA_25p
	
	merge 1:1 hhid sex using `PROWEAI_results_GPI.dta'
	drop _merge
	merge m:m sex using `PROWEAI_results_gender.dta'
	drop _merge

*******************************************
*** Calculate 5DE and PROWEAI score  ***
*******************************************

**** GENERATE KEY VARIABLES ****

	bys sex: gen PROWEAI_3DE=EA_`k'p
	label var PROWEAI_3DE "3DE"

	bys sex: gen disempowered=1-PROWEAI_3DE
	label var disempowered "1-3DE"

	bys sex: gen achieve_emp=1-H_`k'p
	label var achieve_emp "1-H % indiv achieving empowerment"

	bys sex: gen noachieve_emp=H_`k'p
	label var noachieve_emp "H % indiv not achieving empowerment"

	bys sex:  gen mean3DE_unemp=1-A_`k'p
	label var mean3DE_unemp "1-A mean 3DE score for disempowered"

	bys sex:  gen mean_disemp=A_`k'p
	label var mean_disemp "A mean diesmp score 1-3DE for disempowered"

	bys sex: gen achieve_GP= 1-H_GPI
	label var achieve_GP "1-H_GPI % achieving gender parity"

	bys sex: gen noachieve_GP= H_GPI
	label var noachieve_GP "H_GPI % not achieving gender parity"

	bys sex:  gen GPI_gap=ci_gap
	label var GPI_gap "ci_gap gap btwn womens and mens weighted inadequacy scores"
	
	bys sex: gen I_GPI=ci_average
	label var I_GPI "ci_average Average empowerment gap"
	
	gen emp_score = 1-ci
	label var emp_score "Empowerment Score"

	gen empowered = 1-ch_`k'p
	label var empowered "Individual is Empowered"
	cap label drop yesno
	label define yesno 1 "Yes" 0 "No"

	label values empowered yesno
	gen hh_ineq=(W_ci-M_ci) if sex==2
	label var hh_ineq "Intra-household inequality score"

	gen gender_parity = 1-ci_above
	label var gender_parity "Household achieves Gender Parity"
	label values gender_parity yesno

	gen No_obs= sample_r_after 
	gen percent_dataused = sample_r_after/sample_r_before
	
	//Indices of empowerment at individual level
	bys sex: gen IndPROWEAI_3DE=IndEA_25p //added
	bys sex: gen Inddisempowered=1-IndPROWEAI_3DE //added
	bys sex: gen Indachieve_emp=1-IndH_25p //added
	bys sex: gen Indnoachieve_emp=IndH_25p //added
	bys sex:  gen Indmean3DE_unemp=1-IndA_25p //added
	bys sex:  gen Indmean_disemp=IndA_25p //added
	bys sex: gen Indachieve_GP= 1-ci_above //added
	bys sex: gen Indnoachieve_GP= ci_above //added
	bys sex: gen IndI_GPI=ci_gap //added

	*** CALCULATE PROWEAI SCORE ***
	
	gen PROWEAI=(.9*PROWEAI_3DE)+(.1*GPI) if sex==2
	gen IndPROWEAI=(.9*IndPROWEAI_3DE)+(.1*IndGPI) if sex==2 //added
	
	*** CREATE DUAL-HEADED HOUSEHOLD VARIABLE *** 
	
	cap drop temp
	bysort hhid: egen temp = max(n_missing) // This looks at if either respondent in a hh has missing indicators
	gen temp1=1 if sex==1 & temp==0 // Indicator varible for men in HHs where both respondents have all indicators 
	egen dahh = total(temp1) // Count HHs where both respondents have all indicators
	label var dahh "No. of dual adult headed households"
	
// Change the indicators back to 1=adequate 	
foreach var in autonomy_inc selfeff never_violence feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember /*group_inf respect*/ {
	drop `var'
	rename `var'_ndepr `var'
	}
	
*isvar hhid mid sex autonomy_inc selfeff never_violence /*respect*/ feelinputdecagr assetownership credit_accdec incomecontrol work_balance mobility groupmember /*group_inf*/ cont_*  PROWEAI_3DE disempowered achieve_emp noachieve_emp mean3DE_unemp mean_disemp achieve_GP noachieve_GP GPI_gap I_GPI emp_score empowered hh_ineq gender_parity No_obs PROWEAI GPI dahh educ age

*keep `r(varlist)'

*PROWEAI results table - women
*a) original calculation (on aggregate level) for all valid women's responses
sum No_obs PROWEAI_3DE disempowered achieve_emp noachieve_emp mean3DE_unemp mean_disemp GPI achieve_GP noachieve_GP I_GPI PROWEAI if sex==2
*b) alternative calculation (on record level) for all valid women's responses
sum No_obs IndPROWEAI_3DE Inddisempowered Indachieve_emp Indnoachieve_emp Indmean3DE_unemp Indmean_disemp IndGPI Indachieve_GP Indnoachieve_GP IndI_GPI IndPROWEAI if sex==2

*PROWEAI results table - men
*a) original calculation for all valid men's responses
sum No_obs PROWEAI_3DE disempowered achieve_emp noachieve_emp mean3DE_unemp mean_disemp if sex==1
*b) alternative calculation (on record level) for all valid men's responses
sum No_obs IndPROWEAI_3DE Inddisempowered Indachieve_emp Indnoachieve_emp Indmean3DE_unemp Indmean_disemp if sex==1

*Descriptive statistics (incl. standard error) of empowerment score variables
*a) overall empowerment score: 3DE, b) overall disempowerment score, 
*c) empowerment score of disempowered, d) disempowerment score of disempowered
foreach var in IndPROWEAI_3DE Inddisempowered Indmean3DE_unemp Indmean_disemp {
mean `var' if sex==2
mean `var' if sex==1
}

*Descriptive statistics (incl. standard error) of of GPI and PROWEAI
*a) GPI, b) gender gap for women not achieving parity, c) PROWEAI
foreach var in IndGPI IndI_GPI IndPROWEAI{
mean `var' if sex==2 
}


********************************************************************************
						*** SAVE PRO-WEAI DATA *** 
********************************************************************************


save "${projectname}_proweai_calculate", replace 
log close

clear


